
(* This module deals with web transactions (like downloading things) and utilities.

   dependencies:
   - Http module, and all other web protocol implementations *)


module Web = 


  type typehttprequest = GET | POST of string
  type httprequest =
    {
      httphostname: string;
      httppath: string;
      httpport: int;
      httprequestcookies: list string; 
      typerequest: typehttprequest;
      replycallback: (string * (list string)) -> unit; (* content * cookies *)
    }

  type transactionstate = Resolving | Requesting | ToErase
  type transactiontype = HttpRequest of httprequest
  type transaction = 
    {
      mutable currenttcptransaction : option Tcptransaction.transaction; 
      mutable transactionstate : transactionstate;
      transactiontype : transactiontype;
      timeout: int64 * int;
      timecreated : int64 * int;
    }
    

    
  let transactions = ref [] (* current transactions *)
  
let timerDoTasks = ref (0L, 0)
let timerDoTasksInterval = (0L, 200000000) (* in nanoseconds - 200ms *)
let maxconcurrenttransactions = 20




let log msg= 
  let (sec, nsec) = Posix.clock_gettime_monotonic () in 
  printf "[%Li.%09i] %s\n" sec nsec msg


  let escapeUTF8ForXML s =
    let buf = Buffer.build (String.length s * 2) in 
    let f i = 
      let c = s.[i] in 
      if c = '&' || c = '<' || c = '>' then
        Buffer.add_string buf ("&#" ^ (sprintf "%i" (int_of_char c)) ^ ";")
      else
        Buffer.add_char buf c
      in 
    let () = for 0 (String.length s - 1) f in 
    Buffer.contents buf
      
      
  let base64chars = [|
'A';'B';'C';'D';'E';'F';'G';'H';'I';'J';'K';'L';'M';'N';'O';'P';
'Q';'R';'S';'T';'U';'V';'W';'X';'Y';'Z';'a';'b';'c';'d';'e';'f';
'g';'h';'i';'j';'k';'l';'m';'n';'o';'p';'q';'r';'s';'t';'u';'v';
'w';'x';'y';'z';'0';'1';'2';'3';'4';'5';'6';'7';'8';'9';'+';'/'; |]


  let buildBase64Table () = 
    let arr = Array.build 256 in 
    let () = 
     for 0 (Array.length base64chars) 
       (fun i -> let code = int_of_char base64chars[i] in 
                 arr[code] <- i) in 
    arr
    
    
  let encodeBase64 input =
    let len = String.length input in
    let alphabet = buildBase64Table () in 
    let words = (len + 2) / 3 in 
    let padding_len = if len mod 3 = 0 then 0 else 3 - (len mod 3) in
    let output = String.build (words * 4) in
    let get i = if i >= len then 0 else int_of_char input.[i] in
    let f i = 
      let x = get (3 * i + 0) in 
      let y = get (3 * i + 1) in 
      let z = get (3 * i + 2) in
      let n = (x lsl 16) lor (y lsl 8) lor z in
      let a = (n lsr 18) land 0x3f in 
      let b = (n lsr 12) land 0x3f in 
      let c = (n lsr 6) land 0x3f in 
      let d = n land 0x3f in
      let () = output.[4 * i + 0] <- base64chars[a] in  
      let () = output.[4 * i + 1] <- base64chars[b] in 
      let () = output.[4 * i + 2] <- base64chars[c] in 
      let () = output.[4 * i + 3] <- base64chars[d] in 
      ()
     in 
    let () = for  0  (words - 1)  f in 
    
    let () = 
     for  1  padding_len  
       (fun i -> output.[String.length output - i] <- '=') in 
    
    output
  
  
  
let decodeBase64 s = (* garbage in, garbage out / no exception raised *)
  let n = String.length s in 
  let arr = buildBase64Table () in 
  let buf = Buffer.build (String.length s * 2) in 
  let a = ref 0 in 
  let f i = 
    let i1 = arr[int_of_char s.[i * 4]] in 
    let i2 = arr[int_of_char s.[i * 4 + 1]] in 
    let i3 = arr[int_of_char s.[i * 4 + 2]] in 
    let i4 = arr[int_of_char s.[i * 4 + 3]] in 
  
    let () = a := i1 lsl (6 * 3) lor (i2 lsl (6 * 2)) lor (i3 lsl (6 * 1)) lor i4 in 
  
    let c1 = char_of_int (!a lsr (8 * 2)) in 
    let c2 = char_of_int (!a lsr (8 * 1) land 0xff) in 
    let c3 = char_of_int (!a land 0xff) in 
  
    let () = Buffer.add_char buf c1 in 
    let () = if c2 <> '\x00' then Buffer.add_char buf c2 else () in (* handle the padding *)
    if c3 <> '\x00' then Buffer.add_char buf c3 else ()
   in 
  let () = for 0 (n / 4 - 1) f in 
  Buffer.contents buf
  
  




type uri = 
  {
    protocol: string;
    hostname: string;
    port: int;
    path: string;
  }

  

let parse_uri uri = 
  if String.length uri <= 8 || not (String.sub uri 0 7 = "http://" || String.sub uri 0 8 = "https://") then 
    None
  else
    let protocol = if String.sub uri 0 8 = "https://" then "https" else "http" in 
    let lengthprotocol = if protocol = "https" then 8 else 7 in 
    let endhostpost = case String.indexoffrom "/" uri lengthprotocol | Some p -> p | None -> String.length uri end in 
    let host = String.sub uri lengthprotocol (endhostpost - lengthprotocol) in 
    let (hostname, port) = 
      case String.indexof ":" host 
      | Some portpos -> (String.prefix host portpos, String.to_int (String.sub host (portpos + 1) (String.length host - portpos - 1)))
      | None -> (host, if protocol = "https" then 443 else 80) 
      end in 
    Some 
    { protocol = protocol; 
      hostname = hostname; 
      port = port; 
      path = if endhostpost = String.length uri then "/" else String.sub uri endhostpost (String.length uri - endhostpost); }

  
  
  
type get_uri_return = Success of string | Failure
  

(* callback will be called once there is a result *)
let get_uri uri_string callback = 
  case parse_uri uri_string
  | Some uri -> 
     (* TODO: first get the DNS *)
     let () = Dnsresolver.createResolveTransaction uri.hostname Dnsresolver.A ["8.8.4.4"; "8.8.8.8"] in 
     Success ""
  | None -> Failure
  end
  



let closeandclean trans = 
  if trans.transactionstate <> ToErase then 
    (* let () = case trans.currenttlstransaction | None -> () | Some tlstrans ->  trans.currenttlstransaction <- None end in TODO *)
    let () = case trans.currenttcptransaction | None -> () | Some tcptrans -> let () = Tcptransaction.closeandclean tcptrans in trans.currenttcptransaction <- None end in 
    trans.transactionstate <- ToErase
  else
    ()
    
    
let buildrequest req  cookieheader postheaders= 
  case req.typerequest | POST _ -> "POST" | GET -> "GET" end ^ " " ^
  req.httppath ^ " HTTP/1.1\r\n\
Host: " ^ req.httphostname ^ "\r\n\
User-Agent: Mozilla/5.0 (Macintosh; Intel Mac OS X 10.9; rv:47.0) Gecko/20100101 Firefox/47.0\r\n\
Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8\r\n\
Accept-Language: en-US,en;q=0.5\r\n\
Accept-Encoding: gzip, deflate\r\n\
Connection: keep-alive\r\n" ^ cookieheader ^ postheaders ^ "\r\n" ^
case req.typerequest | POST post -> post | GET -> "" end



let createtcptransaction trans remoteaddr request callback = 
      let () = printf "createtcptransaction:\n%s\nin web\n" request in 
      let isreadreadycallback _ = true in 
      let res = Tcptransaction.createTransaction Posix.AF_INET
                                                 (Posix.ADDR_INET(Posix.in_addr_of_string "0.0.0.0", 0)) 
                                                 remoteaddr  
                                                 callback 
                                                 isreadreadycallback in 
      case res  
      | Some tcptrans -> let () = trans.currenttcptransaction <- Some tcptrans in
                         let _ = Tcptransaction.senddata request tcptrans in 
                         trans.transactionstate <- Requesting
      | None -> trans.transactionstate <- ToErase
      end
      
      
let processReceivedData trans tcptrans data lendata = 
  case trans.transactiontype  
  | HttpRequest req -> let () = req.replycallback (data, []) in closeandclean trans
  end 
  
     
let processResolving trans = 
  let (url, request)= 
         case trans.transactiontype  
         | HttpRequest req -> 
           let cookieheader = if req.httprequestcookies = [] then "" else "Cookie: " ^ (String.join req.httprequestcookies "; ") ^ "\r\n" in 
           let postheaders = case req.typerequest | POST post -> sprintf "Content-Type: application/x-www-form-urlencoded\r\nContent-Length: %i\r\n" (String.length post) | GET -> "" end in 
           ({ protocol = case trans.transactiontype | HttpRequest _ -> "http" end;
              hostname = req.httphostname; 
              path = req.httppath; 
              port = req.httpport; }, 
                 buildrequest req cookieheader postheaders) 
         end in 
  case Dnsresolver.getipfromname url.hostname (* hostname may be an ip address *)
  | None -> 
      if not (List.exists (fun tr -> tr.Dnsresolver.hostname = url.hostname) !Dnsresolver.transactions) then 
        Dnsresolver.createResolveTransaction url.hostname Dnsresolver.A ["8.8.4.4"; "8.8.8.8"]
      else
        ()
  | Some ip -> 
      let addr = Posix.ADDR_INET(Posix.in_addr_of_string ip, url.port) in 
      case trans.transactiontype  
      | HttpRequest _ -> createtcptransaction trans addr request (processReceivedData trans)
      end
  end
  
  
  
let createhttptransactionfull protocol hostname port path callback timeout typerequest cookies =
  let () = printf "calling createhttptransactionfull to hostname %s and path %s\n" hostname path in 
  let httprequest = { httphostname = hostname; httppath = path; httpport = port; httprequestcookies = cookies; 
                             typerequest = typerequest; replycallback = callback; } in 
  let transactiontype = HttpRequest httprequest in 
  let trans = { currenttcptransaction = None; transactionstate = Resolving; transactiontype = transactiontype; 
                     timeout = timeout; timecreated = Posix.clock_gettime_monotonic (); } in 
  let () = transactions := trans :: !transactions in 
  trans
  
  
let download protocol hostname path callback timeout = 
  let trans = createhttptransactionfull protocol hostname 80 path callback timeout GET [] in 
  let () = processResolving trans in  (* trick to speed up the process by one loop cycle *)
  trans
    
    
let doTasks t = 
  let () = log "in Web.doTasks" in 
  let diff = Time.diff t !timerDoTasks in 
  if Time.compare diff timerDoTasksInterval > 0 then 
    let () = timerDoTasks := t in 
    let () = List.iter (fun trans -> if Time.compare (Time.diff t trans.timecreated) trans.timeout > 0 && trans.transactionstate <> ToErase then 
                                       case trans.transactiontype  
                                       | HttpRequest req -> let () = req.replycallback ("", []) in closeandclean trans
                                       end
                                     else
                                       ()) !transactions in 
    let () = List.iter processResolving (List.filter (fun trans -> trans.transactionstate = Resolving) !transactions) in 
    transactions := List.filter (fun trans -> trans.transactionstate <> ToErase) !transactions
  else
    ()
  

endmodule
